//
//  ErrorPublisher.swift
//  Do It
//
//  Created by Jim Dovey on 4/9/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

typealias ErrorPublisher = PassthroughSubject<Error,Never>

struct ErrorPublisherEnvironmentKey: EnvironmentKey {
    typealias Value = ErrorPublisher
    static var defaultValue = ErrorPublisher()
}

extension EnvironmentValues {
    var errorPublisher: ErrorPublisher {
        get { self[ErrorPublisherEnvironmentKey.self] }
        set { self[ErrorPublisherEnvironmentKey.self] = newValue }
    }
}

struct ErrorPublishingModifier: ViewModifier {
    @Environment(\.errorPublisher) var subject
    
    struct ErrorInfo: Identifiable {
        var id = UUID()
        var error: Error
        init(_ error: Error) { self.error = error }
    }
    @State private var errorToShow: ErrorInfo? = nil
    
    func body(content: Content) -> some View {
        content
            .onReceive(subject.receive(on: RunLoop.main)) { self.errorToShow = ErrorInfo($0) }
            .alert(item: $errorToShow) { info in
                let title: String
                let message: String?
                
                if let error = info.error as? LocalizedError,
                    let description = error.errorDescription {
                    title = description
                    message = error.failureReason
                }
                else {
                    let error = info.error as NSError
                    title = error.localizedDescription
                    message = error.localizedFailureReason
                }
                
                return Alert(title: Text(title),
                             message: message.map(Text.init(verbatim:)),
                             dismissButton: .default(Text("Dismiss")))
        }
    }
}

extension View {
    func publishingErrors() -> some View {
        modifier(ErrorPublishingModifier())
    }
}
